<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Seosetting;
use App\Models\Apptitle;
use App\Models\Footertext;
use App\Models\Languages;
use App\Models\Translate;
use App\Models\Setting;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Torann\GeoIP\Facades\GeoIP;
use Jenssegers\Agent\Agent;
use App\Models\EmployeeActivity;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Stichoza\GoogleTranslate\GoogleTranslate;

class LanguagesController extends Controller
{
    public function index()
    {
        $this->authorize('Languages Access');
        $title = Apptitle::first();
        $data['title'] = $title;

        $footertext = Footertext::first();
        $data['footertext'] = $footertext;

        $seopage = Seosetting::first();
        $data['seopage'] = $seopage;

        $languages = Languages::get();
        $data['languages'] = $languages;

        foreach ($languages as $language) {

            $path = base_path("lang/{$language->languagecode}.json");
            $translations = json_decode(File::get($path), true);
            $nullCount = 0;

            // Check if translations are an array
            if (is_array($translations)) {
                // Iterate through each group
                foreach ($translations as $group) {
                    // Count null values in the group
                    foreach ($group as $value) {
                        if ($value === null) {
                            $nullCount++;
                        }
                    }
                }
                $language->translates_count = $nullCount;
            }
        }

        return view('admin.languagues.index')->with($data);
    }

    public function create()
    {
        $this->authorize('Languages Create');

        $title = Apptitle::first();
        $data['title'] = $title;

        $footertext = Footertext::first();
        $data['footertext'] = $footertext;

        $seopage = Seosetting::first();
        $data['seopage'] = $seopage;


        return view('admin.languagues.create')->with($data);
    }

    public function edit($lang)
    {

        $this->authorize('Languages Edit');

        $title = Apptitle::first();
        $data['title'] = $title;

        $footertext = Footertext::first();
        $data['footertext'] = $footertext;

        $seopage = Seosetting::first();
        $data['seopage'] = $seopage;

        $language = Languages::findOrFail($lang);
        $data['language'] = $language;

        return view('admin.languagues.edit')->with($data);
    }

    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'languagename' => ['required', 'string', 'max:150'],
            'languagenativename' => ['required', 'string', 'max:150'],
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->all() as $error) {
                // toastr()->error($error);
            }
            return back()->withInput();
        }

        $language = Languages::findOrFail($id);
        if (!$request->has('is_default')) {
            if ($language->languagecode == setting('default_lang')) {
                return back()->with('error', 'Is Default Language');
            }
        }
        $update = Languages::where('id', $id)->update([
            'languagename' => $request->languagename,
            'languagenativename' => $request->languagenativename,
            'is_rtl' => $request->is_rtl ?  '1' : '0',
        ]);
        if ($update) {
            if ($request->has('is_default')) {
                $data['default_lang'] = removeSpaces($language->languagecode);
                $this->updateSettings($data);
            }

            $geolocation = GeoIP::getLocation(request()->getClientIp());
            $agent = new Agent();
            $activity = new EmployeeActivity();
            $activity->user_id = Auth::user()->id;
            $activity->activity_type = 'Language Updated';
            $activity->ip_address = $geolocation->ip;
            $activity->browser = $agent->browser();
            $activity->device = $agent->device();
            $activity->save();

            return redirect()->route('admin.languages.index')->with('success', lang('Updated Successfully', 'alerts'));
        }
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'languagename' => ['required', 'string', 'max:150'],
            'languagenativename' => ['required', 'string', 'max:150'],
            'languagecode' => ['required', 'string', 'max:10', 'min:2', 'unique:languages']
        ]);

        if (!array_key_exists($request->languagecode, languages())) {
            return back()->with('error', lang('Something went wrong please try again', 'alerts'));
        }

        $create = Languages::create([
            'languagename' => $request->languagename,
            'languagenativename' => $request->languagenativename,
            'languagecode' => $request->languagecode,
            'is_rtl' => $request->is_rtl ?  '1' : '0',
        ]);

        $path = base_path("lang/{$create->languagecode}.json");
        $langDirectory = dirname($path);

        // Create directory if it doesn't exist
        if (!File::exists($langDirectory)) {
            File::makeDirectory($langDirectory, 0755, true);
        }
        // Path to the sample language file
        $samplePath = base_path('lang/sample.json');

        // Check if the sample file exists
        if (File::exists($samplePath)) {
            // Read the contents of sample.json
            $sampleContent = File::get($samplePath);

            // Write the contents to the new language file
            File::put($path, $sampleContent);
        } else {
            // Handle the case where sample.json does not exist
            throw new \Exception("Sample language file not found.");
        }

        if ($request->has('is_default')) {
            $data['default_lang'] = removeSpaces($create->languagecode);
            $this->updateSettings($data);
        }

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Language Created';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return redirect()->route('admin.languages.translate', $create->languagecode)->with('success', lang('Created Successfully', 'alerts'));
    }




    /**
     * Display the specified resource.
     *
     * @param  $lang Language code
     * @param  $group Translate group
     * @return \Illuminate\Http\Response
     */
    public function translate(Request $request, $code, $group = null)
    {
        $this->authorize('Languages Translate');

        $language = Languages::where('languagecode', $code)->firstOrFail();
        $path = base_path("lang/{$code}.json");
        $translations = json_decode(File::get($path), true);
        if ($request->input('search')) {
            $q = $request->input('search');
            $searchResults = [];

            // Check if translations are an array
            if (is_array($translations)) {
                // Iterate through each group
                foreach ($translations as $groupName => $group) {
                    // Check if the group is an array
                    if (is_array($group)) {
                        foreach ($group as $key => $value) {
                            // Check if the key or value matches the search keyword
                            if (stripos($key, $q) !== false || (is_string($value) && stripos($value, $q) !== false)) {
                                $searchResults[$key] = $value; // Add matching key and value to results
                            }
                        }
                    }
                }
            }
            $groups = ['Search Results' => collect($searchResults)];
        } elseif ($request->input('filter')) {
            abort_if($request->input('filter') != 'missing', 404);

            $nullTranslations = [];

            if (is_array($translations)) {
                // Iterate through each group
                foreach ($translations as $groupName => $group) {
                    // Initialize an array for null keys in this group
                    $nullKeys = [];

                    // Check if the group is an array
                    if (is_array($group)) {
                        foreach ($group as $key => $value) {
                            if ($value === null) {
                                $nullKeys[$key] = null; // Collect keys with null values
                            }
                        }
                    }

                    // Add the null keys to the nullTranslations array for this group
                    if (!empty($nullKeys)) {
                        $nullTranslations[$groupName] = $nullKeys;
                    }
                }
            }
            $groups = collect($nullTranslations);
        } else {
            $groups = collect($translations);
        }
        $translates_count = 0;

        if (is_array($translations)) {
            foreach ($translations as $group) {
                foreach ($group as $value) {
                    if ($value === null) {
                        $translates_count++;
                    }
                }
            }
        }
        //     $groups = Translate::where('lang_code', $language->languagecode)->select('group_langname')->groupBy('group_langname')->orderBy('id', 'ASC')->get();
        //     if ($group != null) {
        //         $group = str_replace('-', ' ', $group);
        //         $translates = Translate::where('lang_code', $language->languagecode)->where('group_langname', $group)->get();
        //         abort_if($translates->count() < 1, 404);
        //         $active = $group;
        //     } else {
        //         $translates = Translate::where('lang_code', $language->languagecode)->where('group_langname', 'general')->get();
        //         $active = "general";
        //     }
        // }
        // $translates_count = Translate::where('lang_code', $language->languagecode)->where('value', null)->count();

        $title = Apptitle::first();
        $data['title'] = $title;

        $footertext = Footertext::first();
        $data['footertext'] = $footertext;

        $seopage = Seosetting::first();
        $data['seopage'] = $seopage;

        return view('admin.languagues.translate', [
            // 'active' => $active,
            // 'translates' => $translates,
            'groups' => $groups,
            'language' => $language,
            'translates_count' => $translates_count,
        ])->with($data);
    }


    /**
     * Update the translate.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Language  $language
     * @return \Illuminate\Http\Response
     */
    public function translateUpdate(Request $request, $id)
    {
        $language = Languages::where('id', $id)->first();
        if ($language == null) {
            return redirect(url('admin/languages'));
        }
        $path = base_path("lang/{$language->languagecode}.json");
        if (File::exists($path)) {
            $translations = json_decode(File::get($path), true);
        } else {
            return back()->with('error', lang('Translation file not found.', 'alerts'));
        }

        foreach ($request->except('_token') as $key => $value) {
            $cleanKey = str_replace('$', '.', str_replace('_', ' ', $key));
            foreach ($translations as $group => $groupData) {
                if (array_key_exists($cleanKey, $groupData)) {
                    $translations[$group][$cleanKey] = $value;
                }
            }
        }
        File::put($path, json_encode($translations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Language Translation Updated';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return back()->with('success', lang('Updated Successfully', 'alerts'));
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  $id language ID
     * @return \Illuminate\Http\Response
     */
    public function setDefault($id)
    {
        $language = Languages::find(decrypt($id));

        if ($language != null) {
            if (setting('default_lang') == $language->languagecode) {
                return back()->with('error', lang('Language already marked as default', 'alerts'));
            } else {
                $data['default_lang'] = $language->languagecode;
                $this->updateSettings($data);

                $geolocation = GeoIP::getLocation(request()->getClientIp());
                $agent = new Agent();
                $activity = new EmployeeActivity();
                $activity->user_id = Auth::user()->id;
                $activity->activity_type = 'Default language Language';
                $activity->ip_address = $geolocation->ip;
                $activity->browser = $agent->browser();
                $activity->device = $agent->device();
                $activity->save();

                return back()->with('success', lang('Default language has updated Successfully', 'alerts'));
            }
        } else {
            return back()->with('error', lang('language not exists', 'alerts'));
        }
    }


    /**
     *  Settings Save/Update.
     *
     * @return \Illuminate\Http\Response
     */
    private function updateSettings($data)
    {

        foreach ($data as $key => $val) {
            $setting = Setting::where('key', $key);
            if ($setting->exists())
                $setting->first()->update(['value' => $val]);
        }
    }



    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Language  $language
     * @return \Illuminate\Http\Response
     */
    public function destroy($language)
    {
        $language = Languages::find($language);

        if ($language->languagecode == setting('default_lang')) {

            return response()->json(['error' => lang('Default language cannot be deleted', 'alerts')]);
        }

        $path = base_path("lang/{$language->languagecode}.json");
        if (File::exists($path)) {
            File::delete($path);
        }

        $language->delete();

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Language Deleted';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => lang('Deleted Successfully', 'alerts')]);
    }

    public function autoTranslate(Request $request)
    {
        try {
            $source_language = 'en';
            $target_language = $request->lang;

            $path = base_path("lang/{$request->lang}.json");

            $jsonContent = json_decode(File::get($path), true);

            $translatedContent = [];
            $translate = new GoogleTranslate();
            $translate->setSource($source_language);
            $translate->setTarget($target_language);
            foreach ($jsonContent as $key => $keyvalue) {
                foreach ($keyvalue as $subKey => $value) {
                    $translatedText = $translate->translate($subKey);
                    $translatedContent[$key][$subKey] = $translatedText;
                }
            }
            $jsonData = json_encode($translatedContent, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

            // Write the JSON data back to the file
            File::put($path, $jsonData);

            return Response::json(['success' => lang('Translations updated successfully.')]);
        } catch (\Throwable $th) {
            return Response::json(['error' => $th->getMessage()]);
        }
    }
}
